document.addEventListener('DOMContentLoaded', () => {
    const API_TOKEN = 'ImavcR8ePwOIEkXh';

    // --- DOM Elements ---
    const appContainer = document.getElementById('app-container');
    const chatTabsContainer = document.getElementById('chat-tabs');
    const chatArea = document.getElementById('chat-area');
    const chatMessages = document.getElementById('chat-messages');
    const welcomeScreen = document.getElementById('welcome-screen');
    const promptInput = document.getElementById('prompt-input');
    const sendButton = document.getElementById('send-button');
    const newChatButton = document.getElementById('new-chat-button');
    const webSearchToggle = document.getElementById('web-search-toggle');
    const fileAttachBtn = document.getElementById('file-attach-btn');
    const attachmentPreview = document.getElementById('attachment-preview');
    const attachmentThumbnail = document.getElementById('attachment-thumbnail');
    const removeAttachmentBtn = document.getElementById('remove-attachment-btn');
    
    // --- State ---
    let storeData = {};
    let activeChatId = null;
    let isWebSearchActive = false;

    function adjustHexColor(hex, amount) {
        let r = parseInt(hex.slice(1, 3), 16);
        let g = parseInt(hex.slice(3, 5), 16);
        let b = parseInt(hex.slice(5, 7), 16);
        r = Math.max(0, Math.min(255, r + amount));
        g = Math.max(0, Math.min(255, g + amount));
        b = Math.max(0, Math.min(255, b + amount));
        return `rgb(${r}, ${g}, ${b})`;
    }

    function hexToRgba(hex, alpha) {
        const r = parseInt(hex.slice(1, 3), 16);
        const g = parseInt(hex.slice(3, 5), 16);
        const b = parseInt(hex.slice(5, 7), 16);
        return `rgba(${r}, ${g}, ${b}, ${alpha})`;
    }

    window.electronAPI.invoke('get-store-data').then(data => {
        updateState(data);
    });

    function updateState(data) {
        storeData = data;
        if (!storeData.chatSessions || storeData.chatSessions.length === 0) {
            return createNewChat();
        }
        activeChatId = storeData.activeChatId && storeData.chatSessions.some(s => s.id === storeData.activeChatId) 
            ? storeData.activeChatId 
            : storeData.chatSessions[0]?.id;
        
        if (!activeChatId) return createNewChat();
        
        applySettings(storeData.settings);
        renderAll();
    }

    function applySettings(settings) {
        const { tintColor, windowOpacity, userBubbleColor, aiBubbleColor, activeTabColor } = settings;
        
        const root = document.documentElement;
        root.style.setProperty('--tint-color', hexToRgba(tintColor, windowOpacity));
        root.style.setProperty('--bg-secondary', adjustHexColor(tintColor, 20));
        root.style.setProperty('--bg-tertiary', adjustHexColor(tintColor, 30));
        root.style.setProperty('--bg-hover', adjustHexColor(tintColor, 40));
        root.style.setProperty('--border-color', adjustHexColor(tintColor, 50));
        
        root.style.setProperty('--user-bubble-color', userBubbleColor);
        root.style.setProperty('--ai-bubble-color', aiBubbleColor);
        root.style.setProperty('--active-tab-color', activeTabColor);
    }

    function renderAll() {
        renderTabs();
        renderActiveChat();
        renderIcons();
        renderAttachment();
    }

    function renderIcons() {
        newChatButton.innerHTML = icons.plus;
        document.querySelector('#settings-button').innerHTML = icons.settings;
        webSearchToggle.innerHTML = icons.globe;
        fileAttachBtn.innerHTML = icons.paperclip;
        sendButton.innerHTML = icons.send;
        removeAttachmentBtn.innerHTML = icons.x;
    }

    function renderTabs() {
        chatTabsContainer.innerHTML = '';
        storeData.chatSessions.forEach(session => {
            const tab = document.createElement('div');
            tab.className = 'chat-tab';
            tab.dataset.id = session.id;
            tab.classList.toggle('active', session.id === activeChatId);
            const titleSpan = document.createElement('span');
            titleSpan.textContent = session.title;
            tab.appendChild(titleSpan);
            const closeBtn = document.createElement('button');
            closeBtn.className = 'close-tab-btn';
            closeBtn.innerHTML = icons.x;
            closeBtn.onclick = (e) => { e.stopPropagation(); deleteChat(session.id); };
            tab.appendChild(closeBtn);
            tab.onclick = () => switchActiveChat(session.id);
            chatTabsContainer.appendChild(tab);
        });
        chatTabsContainer.scrollLeft = chatTabsContainer.scrollWidth;
    }

    function renderActiveChat() {
        const activeChat = getActiveChat();
        chatMessages.innerHTML = '';
        if (activeChat && activeChat.messages.length > 0) {
            welcomeScreen.style.display = 'none';
            chatMessages.style.display = 'flex';
            activeChat.messages.forEach(msg => addMessageToDOM(msg.content, msg.sender, msg.type));
        } else {
            welcomeScreen.style.display = 'flex';
            chatMessages.style.display = 'none';
        }
        autoScroll();
    }

    function renderAttachment() {
        const activeChat = getActiveChat();
        if (activeChat && activeChat.attachedImage) {
            attachmentThumbnail.src = activeChat.attachedImage;
            attachmentPreview.classList.remove('hidden');
        } else {
            attachmentPreview.classList.add('hidden');
        }
    }
    
    const getActiveChat = () => storeData.chatSessions.find(s => s.id === activeChatId);

    function switchActiveChat(id) {
        activeChatId = id;
        window.electronAPI.send('set-store-data', { activeChatId: id });
    }

    function createNewChat() {
        const newId = `chat-${Date.now()}`;
        const newTitle = new Date().toLocaleString('en-US', {
            month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit'
        });
        const newSessions = [...(storeData.chatSessions || []), { id: newId, title: newTitle, messages: [], attachedImage: null }];
        window.electronAPI.send('set-store-data', { chatSessions: newSessions, activeChatId: newId });
    }
    
    function deleteChat(id) {
        const newSessions = storeData.chatSessions.filter(s => s.id !== id);
        let newActiveId = activeChatId;
        if (activeChatId === id) {
            newActiveId = newSessions[newSessions.length - 1]?.id || null;
        }
        if (newSessions.length === 0) {
            return createNewChat();
        }
        window.electronAPI.send('set-store-data', { chatSessions: newSessions, activeChatId: newActiveId });
    }
    
    function updateCurrentChat(data) {
        const newSessions = storeData.chatSessions.map(s => 
            s.id === activeChatId ? { ...s, ...data } : s
        );
        window.electronAPI.send('set-store-data', { chatSessions: newSessions });
    }
    
    function addMessageToCurrentChat(message) {
        const activeChat = getActiveChat();
        if (!activeChat) return;
        const updatedMessages = [...activeChat.messages, message];
        updateCurrentChat({ messages: updatedMessages });
    }

    const autoScroll = () => { 
        chatArea.scrollTop = chatArea.scrollHeight;
    };

    const parseMarkdown = (text) => {
        let html = text
            .replace(/</g, '<')
            .replace(/>/g, '>');
        html = html
            .replace(/```([\s\S]*?)```/g, (match, code) => `<pre><code>${code.trim()}</code></pre>`)
            .replace(/\[([^\]]+)\]\(([^)]+)\)/g, '<a href="$2" class="markdown-link" target="_blank">$1</a>')
            .replace(/^## (.*$)/gim, '<h2>$1</h2>')
            .replace(/^# (.*$)/gim, '<h1>$1</h1>')
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/__(.*?)__/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/_(.*?)_/g, '<em>$1</em>')
            .replace(/`([^`]+)`/g, '<code>$1</code>')
            .replace(/^\s*([*-])\s+(.*)/gim, '<ul><li>$2</li></ul>')
            .replace(/<\/ul>\n<ul>/gim, '');
        return html.replace(/\n/g, '<br>')
                   .replace(/<br>\s*<(h1|h2|ul|li|pre)/g, '<$1')
                   .replace(/<\/(h1|h2|ul|li|pre)>\s*<br>/g, '</$1>');
    };

    const addMessageToDOM = (content, sender, type = 'text') => {
        if (welcomeScreen.style.display !== 'none') {
            welcomeScreen.style.display = 'none';
            chatMessages.style.display = 'flex';
        }
        const msgWrapper = document.createElement('div');
        msgWrapper.className = `message-wrapper message-${sender}`;
        const msgEl = document.createElement('div');
        msgEl.className = 'chat-bubble';
        const actionsContainer = document.createElement('div');
        actionsContainer.className = 'message-actions';
        if (type === 'image') {
            const img = document.createElement('img');
            img.src = content;
            img.onload = autoScroll;
            const saveBtn = document.createElement('button');
            saveBtn.className = 'action-btn';
            saveBtn.innerHTML = icons.download;
            saveBtn.onclick = () => window.electronAPI.send('save-image-to-disk', content);
            actionsContainer.appendChild(saveBtn);
            msgEl.appendChild(img);
        } else {
            msgEl.innerHTML = parseMarkdown(content);
            if (sender === 'ai') {
                const copyBtn = document.createElement('button');
                copyBtn.className = 'action-btn';
                copyBtn.innerHTML = icons.copy;
                copyBtn.onclick = () => navigator.clipboard.writeText(content);
                actionsContainer.appendChild(copyBtn);
            }
        }
        msgWrapper.appendChild(msgEl);
        if(actionsContainer.hasChildNodes()) msgWrapper.appendChild(actionsContainer);
        chatMessages.appendChild(msgWrapper);
        autoScroll();
        return msgWrapper;
    };
    
    // --- THIS IS THE KEY FIX AREA ---
    function handleInputCommand(prompt) {
        // First, just add the user's message to the screen for immediate feedback.
        // DO NOT save it to the state yet.
        addMessageToDOM(prompt, 'user');

        if (prompt.toLowerCase().startsWith('/image')) {
            const imagePrompt = prompt.split(' ').slice(1).join(' ').trim();
            if (imagePrompt) {
                // The image flow correctly waits to save state until the end.
                promptForResolution(imagePrompt, prompt);
            } else {
                addMessageToDOM("Please provide a prompt for the image. e.g., `/image a cat`", 'ai');
            }
        } else {
            // REMOVED the state update from here.
            // We now pass the prompt to handleTextRequest, which will handle state saving.
            handleTextRequest(prompt);
        }
    }

    async function generateImage(prompt, width, height, originalFullPrompt) {
        const thinkingMessageWrapper = addMessageToDOM('', 'ai');
        const thinkingBubble = thinkingMessageWrapper.querySelector('.chat-bubble');
        thinkingBubble.classList.add('thinking');
        
        for (let i = 0; i < 3; i++) thinkingBubble.appendChild(document.createElement('span'));

        const timerText = document.createElement('span');
        timerText.className = 'thinking-timer';
        timerText.textContent = '0s';
        thinkingBubble.appendChild(timerText);

        let seconds = 0;
        const timerInterval = setInterval(() => {
            seconds++;
            timerText.textContent = `${seconds}s`;
        }, 1000);

        // This function saves BOTH user and AI messages after completion.
        const finishGeneration = (aiMessage) => {
            clearInterval(timerInterval);
            thinkingMessageWrapper.remove();
            
            const userMessage = { sender: 'user', content: originalFullPrompt, type: 'text' };
            const activeChat = getActiveChat();
            // This is the correct pattern: update state with a complete transaction.
            updateCurrentChat({ messages: [...activeChat.messages, userMessage, aiMessage] });
        };

        try {
            const model = storeData.settings.imageModel || 'flux';
            const encodedPrompt = encodeURIComponent(prompt);
            const url = `https://image.pollinations.ai/prompt/${encodedPrompt}?width=${width}&height=${height}&model=${model}&token=${API_TOKEN}&nologo=true&private=true&enhance=true&safe=false`;
            
            const img = new Image();
            img.onload = () => {
                finishGeneration({ sender: 'ai', content: url, type: 'image' });
            };
            img.onerror = () => {
                finishGeneration({ sender: 'ai', content: 'Image generation failed. Please try again.', type: 'text' });
            };
            img.src = url;

        } catch (error) {
            finishGeneration({ sender: 'ai', content: `Image Gen Error: ${error.message}`, type: 'text' });
        }
    }

    // --- THIS IS THE SECOND KEY FIX AREA ---
    async function handleTextRequest(prompt) {
        const thinkingMessageWrapper = addMessageToDOM('', 'ai');
        const thinkingBubble = thinkingMessageWrapper.querySelector('.chat-bubble');
        thinkingBubble.classList.add('thinking');
        for(let i=0; i<3; i++) thinkingBubble.appendChild(document.createElement('span'));
        
        const timerText = document.createElement('span');
        timerText.className = 'thinking-timer';
        timerText.textContent = '0s';
        thinkingBubble.appendChild(timerText);

        let seconds = 0;
        const timerInterval = setInterval(() => {
            seconds++;
            timerText.textContent = `${seconds}s`;
        }, 1000);
        
        try {
            let model = isWebSearchActive ? 'searchgpt' : storeData.settings.textModel;
            let body;
            const activeChat = getActiveChat();
            
            if (activeChat.attachedImage) {
                model = 'mistral'; 
                body = JSON.stringify({
                    model,
                    messages: [{ 
                        role: "user", 
                        content: [
                            { type: "text", text: prompt }, 
                            { type: "image_url", image_url: { "url": activeChat.attachedImage } }
                        ] 
                    }]
                });
            } else {
                body = JSON.stringify({ model, messages: [{ role: "user", content: prompt }] });
            }

            const response = await fetch('https://text.pollinations.ai/openai', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'Authorization': `Bearer ${API_TOKEN}`},
                body: body
            });

            if (!response.ok) throw new Error(`${response.status}: ${await response.text()}`);
            const result = await response.json();
            const aiResponse = result.choices[0].message.content;
            
            clearInterval(timerInterval);
            thinkingMessageWrapper.remove();
            
            // Following the image generator's pattern:
            // Create BOTH message objects here.
            const userMessage = { sender: 'user', content: prompt, type: 'text' };
            const aiMessage = { sender: 'ai', content: aiResponse, type: 'text' };
            
            // And save them to state together.
            const currentChat = getActiveChat();
            updateCurrentChat({ messages: [...currentChat.messages, userMessage, aiMessage] });

        } catch (error) {
            clearInterval(timerInterval);
            const errorText = `Error: ${error.message}`;
            // Just update the temporary bubble with the error. Don't save to history.
            thinkingBubble.classList.remove('thinking');
            thinkingBubble.innerHTML = parseMarkdown(errorText);
        }
    }

    function promptForResolution(imagePrompt, originalFullPrompt) {
        const resContainer = document.createElement('div');
        resContainer.className = 'message-wrapper message-ai';
        const bubble = document.createElement('div');
        bubble.className = 'chat-bubble resolution-picker';
        bubble.innerHTML = `<span>Pick a resolution:</span>`;
        const resolutions = ['1024x1024', '1280x768', '768x1280', '768x768'];
        resolutions.forEach(res => {
            const btn = document.createElement('button');
            btn.textContent = res;
            btn.onclick = () => {
                resContainer.remove();
                const [width, height] = res.split('x');
                generateImage(imagePrompt, width, height, originalFullPrompt);
            };
            bubble.appendChild(btn);
        });
        resContainer.appendChild(bubble);
        chatMessages.appendChild(resContainer);
        autoScroll();
    }

    // --- Event Listeners ---
    sendButton.addEventListener('click', () => {
        const prompt = promptInput.value.trim();
        if (prompt) {
            promptInput.value = '';
            handleInputCommand(prompt);
            promptInput.style.height = 'auto';
        }
    });
    promptInput.addEventListener('keydown', (e) => { if (e.key === 'Enter' && !e.shiftKey) { e.preventDefault(); sendButton.click(); }});
    promptInput.addEventListener('input', () => {
        promptInput.style.height = 'auto';
        promptInput.style.height = `${promptInput.scrollHeight}px`;
    });
    newChatButton.addEventListener('click', createNewChat);
    
    document.querySelector('.prompt-suggestions').addEventListener('click', (e) => {
        if(e.target.classList.contains('suggestion-btn')) {
            promptInput.value = e.target.textContent;
            sendButton.click();
        }
    });
    document.body.addEventListener('click', e => {
        const link = e.target.closest('a.markdown-link');
        if (link && link.href) {
            e.preventDefault();
            window.electronAPI.send('open-external-link', link.href);
        }
    });
    webSearchToggle.addEventListener('click', () => {
        isWebSearchActive = !isWebSearchActive;
        webSearchToggle.classList.toggle('active', isWebSearchActive);
    });
    fileAttachBtn.addEventListener('click', async () => {
        const imageDataUrl = await window.electronAPI.invoke('open-file-dialog');
        if (imageDataUrl) {
            updateCurrentChat({ attachedImage: imageDataUrl });
        }
    });
    removeAttachmentBtn.addEventListener('click', () => {
        updateCurrentChat({ attachedImage: null });
    });
    document.getElementById('settings-button').addEventListener('click', () => window.electronAPI.send('show-settings'));
    
    window.electronAPI.on('store-updated', (data) => updateState(data));
    window.electronAPI.on('toggle-slide', () => {
        appContainer.classList.toggle('is-slid-out');
        window.electronAPI.send('set-window-interactivity', !appContainer.classList.contains('is-slid-out'));
    });
});